global class PublishDraftArticlesBatch implements Database.Batchable<sObject>, Database.Stateful {
    
    // STATEFUL VARIABLES: Track progress across chunks
    global Integer recordsProcessed = 0;
    global Integer recordsPublished = 0;
    global Integer recordsFailed = 0;
    global String errorLog = '';

    // 1. START: Collect ALL Draft Articles
    global Database.QueryLocator start(Database.BatchableContext BC) {
        // We need the KnowledgeArticleId for the publishing service
        // We verify PublishStatus is Draft to avoid errors trying to publish already online articles
        String query = 'SELECT Id, KnowledgeArticleId, Title FROM Knowledge__kav WHERE PublishStatus = \'Draft\'';
        System.debug('### BATCH STARTED: Querying for Draft Articles to Publish...');
        return Database.getQueryLocator(query);
    }

    // 2. EXECUTE: Process records in chunks
    global void execute(Database.BatchableContext BC, List<Knowledge__kav> scope) {
        
        for (Knowledge__kav k : scope) {
            recordsProcessed++;
            try {
                // *** THE CORE PUBLISH COMMAND ***
                // Param 1: The KnowledgeArticleId (Master ID)
                // Param 2: flagAsNew (true = shows as a new version in history)
                KbManagement.PublishingService.publishArticle(k.KnowledgeArticleId, true);
                
                recordsPublished++;
                
            } catch (Exception e) {
                recordsFailed++;
                String err = 'FAIL on "' + k.Title + '": ' + e.getMessage();
                System.debug(err);
                
                if (errorLog.length() < 30000) {
                    errorLog += err + '\n';
                }
            }
        }
    }

    // 3. FINISH: Summary
    global void finish(Database.BatchableContext BC) {
        System.debug('### BATCH COMPLETE ###');
        System.debug('Total Processed: ' + recordsProcessed);
        System.debug('Total Published: ' + recordsPublished);
        System.debug('Total Failed: '    + recordsFailed);
        
        if (recordsFailed > 0) {
            System.debug('### ERROR SUMMARY ###');
            System.debug(errorLog);
        }
    }
}