global class ArchivePublishArticlesBatch implements Database.Batchable<sObject>, Database.Stateful {
    
    // STATEFUL VARIABLES: Track progress across chunks
    global Integer recordsProcessed = 0;
    global Integer recordsArchived = 0;
    global Integer recordsFailed = 0;
    global String errorLog = '';

    // 1. START: Collect ALL Online (Published) Articles
    global Database.QueryLocator start(Database.BatchableContext BC) {
        // We verify PublishStatus is 'Online' to avoid errors trying to archive drafts or already archived items
        String query = 'SELECT Id, KnowledgeArticleId, Title FROM Knowledge__kav WHERE PublishStatus = \'Online\'';
        System.debug('### BATCH STARTED: Querying for Online Articles to Archive...');
        return Database.getQueryLocator(query);
    }

    // 2. EXECUTE: Process records in chunks
    global void execute(Database.BatchableContext BC, List<Knowledge__kav> scope) {
        
        for (Knowledge__kav k : scope) {
            recordsProcessed++;
            try {
                // *** THE CORE ARCHIVE COMMAND ***
                // Param 1: The KnowledgeArticleId (Master ID)
                // Param 2: null (Required by Salesforce API for this method)
                KbManagement.PublishingService.archiveOnlineArticle(k.KnowledgeArticleId, null);
                
                recordsArchived++;
                
            } catch (Exception e) {
                recordsFailed++;
                String err = 'FAIL on "' + k.Title + '": ' + e.getMessage();
                System.debug(err);
                
                if (errorLog.length() < 30000) {
                    errorLog += err + '\n';
                }
            }
        }
    }

    // 3. FINISH: Summary
    global void finish(Database.BatchableContext BC) {
        System.debug('### BATCH COMPLETE ###');
        System.debug('Total Processed: ' + recordsProcessed);
        System.debug('Total Archived: '  + recordsArchived);
        System.debug('Total Failed: '    + recordsFailed);
        
        if (recordsFailed > 0) {
            System.debug('### ERROR SUMMARY ###');
            System.debug(errorLog);
        }
    }
}