global class DeleteDraftArticlesBatch implements Database.Batchable<sObject>, Database.Stateful {
    
    // STATEFUL VARIABLES: These keep their value across the different "chunks" of data
    global Integer recordsProcessed = 0;
    global Integer recordsDeleted = 0;
    global Integer recordsFailed = 0;
    global String errorLog = '';

    // 1. START: Collect ALL Draft Articles
    global Database.QueryLocator start(Database.BatchableContext BC) {
        // We query KnowledgeArticleId because the Delete service requires it (not the record ID)
        String query = 'SELECT Id, KnowledgeArticleId, Title, UrlName FROM Knowledge__kav WHERE PublishStatus = \'Draft\'';
        System.debug('### BATCH STARTED: Querying for Draft Articles...');
        return Database.getQueryLocator(query);
    }

    // 2. EXECUTE: Process records in chunks (e.g., 50 at a time)
    global void execute(Database.BatchableContext BC, List<Knowledge__kav> scope) {
        
        List<Knowledge__kav> deletedRecords = new List<Knowledge__kav>();
        
        for (Knowledge__kav k : scope) {
            recordsProcessed++;
            try {
                // *** THE CORE DELETE COMMAND ***
                // Must use KbManagement service, not standard delete DML
                KbManagement.PublishingService.deleteDraftArticle(k.KnowledgeArticleId);
                
                recordsDeleted++;
                deletedRecords.add(k);
                
            } catch (Exception e) {
                recordsFailed++;
                // Capture the error and the article Title for feedback
                String err = 'FAIL on "' + k.Title + '": ' + e.getMessage();
                System.debug(err);
                
                // Keep a running log of errors (truncated to avoid string limits)
                if (errorLog.length() < 30000) {
                    errorLog += err + '\n';
                }
            }
        }
        
        // EMPTY RECYCLE BIN
        // This is critical to free up the URL Names for your migration
        if (!deletedRecords.isEmpty()) {
            try {
                Database.emptyRecycleBin(deletedRecords);
            } catch (Exception binError) {
                System.debug('Recycle Bin Error: ' + binError.getMessage());
            }
        }
    }

    // 3. FINISH: Runs once at the very end to give you a summary
    global void finish(Database.BatchableContext BC) {
        System.debug('### BATCH COMPLETE ###');
        System.debug('Total Processed: ' + recordsProcessed);
        System.debug('Total Deleted: '   + recordsDeleted);
        System.debug('Total Failed: '    + recordsFailed);
        
        if (recordsFailed > 0) {
            System.debug('### ERROR SUMMARY ###');
            System.debug(errorLog);
        }
        
        // OPTIONAL: Send an email to yourself with the results
        // Uncomment the lines below if you want an email report
        /*
        Messaging.SingleEmailMessage mail = new Messaging.SingleEmailMessage();
        String[] toAddresses = new String[] {UserInfo.getUserEmail()};
        mail.setToAddresses(toAddresses);
        mail.setSubject('Knowledge Batch Delete Results');
        mail.setPlainTextBody('Batch Complete.\nDeleted: ' + recordsDeleted + '\nFailed: ' + recordsFailed + '\n\nErrors:\n' + errorLog);
        Messaging.sendEmail(new Messaging.SingleEmailMessage[] { mail });
        */
    }
}