global class DraftArchiveArticlesBatch implements Database.Batchable<sObject>, Database.Stateful {
    
    // STATEFUL VARIABLES: Track progress across chunks
    global Integer recordsProcessed = 0;
    global Integer recordsRestoredToDraft = 0;
    global Integer recordsFailed = 0;
    global String errorLog = '';

    // 1. START: Collect ALL Archived Articles
    global Database.QueryLocator start(Database.BatchableContext BC) {
        // We query for 'Archived' status. 
        // We also check IsLatestVersion = true to ensure we are only grabbing the most recent archived copy
        // (in case multiple archived versions exist in history).
        String query = 'SELECT Id, KnowledgeArticleId, Title FROM Knowledge__kav WHERE PublishStatus = \'Archived\' AND IsLatestVersion = true';
        System.debug('### BATCH STARTED: Querying for Archived Articles to Restore to Draft...');
        return Database.getQueryLocator(query);
    }

    // 2. EXECUTE: Process records in chunks
    global void execute(Database.BatchableContext BC, List<Knowledge__kav> scope) {
        
        for (Knowledge__kav k : scope) {
            recordsProcessed++;
            try {
                // *** THE CORE RESTORE COMMAND ***
                // Param 1: The KnowledgeArticleId (Master ID)
                // This creates a new Draft version based on the Archived article.
                KbManagement.PublishingService.editArchivedArticle(k.KnowledgeArticleId);
                
                recordsRestoredToDraft++;
                
            } catch (Exception e) {
                recordsFailed++;
                String err = 'FAIL on "' + k.Title + '": ' + e.getMessage();
                System.debug(err);
                
                if (errorLog.length() < 30000) {
                    errorLog += err + '\n';
                }
            }
        }
    }

    // 3. FINISH: Summary
    global void finish(Database.BatchableContext BC) {
        System.debug('### BATCH COMPLETE ###');
        System.debug('Total Processed: ' + recordsProcessed);
        System.debug('Total Restored to Draft: ' + recordsRestoredToDraft);
        System.debug('Total Failed: '    + recordsFailed);
        
        if (recordsFailed > 0) {
            System.debug('### ERROR SUMMARY ###');
            System.debug(errorLog);
        }
    }
}